%% demo_l1.m
% Tests the L1 solvers on randomly-generated instances.
% SIZES(i,:) -> [m,n]
rng(42); % for reproducibility
SIZES = [...
    [1200, 100];
    [1200, 200];
    [1200, 300];
    [1200, 400];
    [1200, 500];
    [1200, 600];
    [1200, 700];
    [1200, 800];
    [1200, 900];
    [1200, 1000];
    [1200, 1100];
    [1200, 1200];
];

%% Run Demo
% Run L1-norm solver on random problem instances, comparing against cvx
% output as a sanity check.
disp('==== Demo-ing L1 simple ====');
tic_all = tic;
durs = []; % durs(i) -> float duration
durs_cvx = [];
for i=1:size(SIZES, 1)
    m = SIZES(i,1);
    n = SIZES(i,2);
    fprintf('[Problem %d/%d] m=%d n=%d\n', i, size(SIZES,1),m,n);
    % Generate random problem instance
    P = randn(m, n);
    q = randn(m, 1);
    % Run solver
    tic;
    u = l1(P, q, 'Verbose', false);
    optval = norm(P*u + q, 1);
    dur = toc;
    fprintf('  optval: %.8f (%.4f secs)\n', optval, dur);
    % Compare against cvx
    tic_cvx= tic;
    [u, cvx_optval, cvx_status] = l1_cvx(P, q);
    dur_cvx = toc(tic_cvx);
    fprintf('  cvx_optval is: %f (%.4f secs)\n', cvx_optval, dur_cvx);

    optval_diff = abs(cvx_optval - optval);
    fprintf('    |cvx_optval - optval| = %f\n', optval_diff);
    if (optval_diff >= 0.001)
        disp('==== DANGER! High discrepancy between optval, cvx_optval');
    end
    durs(i) = dur;
    durs_cvx(i) = dur_cvx;
end
fprintf('==== Finished Demo. (%.4fs) ====\n', toc(tic_all));

%% Plot timing between my solver and cvx
figure;
hold on;
h1 = plot(SIZES(:,2), durs, 'b.-', 'MarkerSize', 12);
h2 = plot(SIZES(:,2), durs_cvx, 'r.-', 'MarkerSize', 12);
grid on;
legend([h1, h2], {'My Solver', 'cvx'});
xlabel('M (nb constraints)');
ylabel('Time (seconds)');
title('Execution Speed Comparison (N=1200)');
set(gcf, 'color', 'white');
% export_fig() is from: 
%   https://github.com/altmany/export_fig
% export_fig('./compare.png', '-m2.5', '-native'); % save figure to .png file

%% Show difference between L1 norm and L2 norm approximation
% L1 norm results in sparse reconstruction error.
% L2 norm results in dense reconstruction error, but overall better fit.
m = 80; n = 60;
rng(42); % for reproducibility
P = randn(m, n);
q = randn(m, 1);
% L1
[u1] = l1(P,q);
qErr1 = abs(P*u1 + q);
err1 = sum((P*u1+q).^2)/length(q); % MSE (mean-squared error)
% L2
u2 = P\(-q);
qErr2 = abs(P*u2 + q);
err2 = sum((P*u2+q).^2)/length(q); % MSE error

%
ymax = max(max(qErr1), max(qErr2));
C_ = 1.25; % ylim padding
xlim_ = [0, m+1];
ylim_ = [0, ymax*C_];
figure;
subplot(2,1,1);
bar(qErr1);
title(sprintf('Reconstruction Error (L1 norm)'));
xlim(xlim_);
ylim(ylim_);
grid on;
subplot(2,1,2);
bar(qErr2);
title(sprintf('Reconstruction Error (L2 norm)'));
xlim(xlim_);
ylim(ylim_);
grid on;
set(gcf, 'color', 'white');
fprintf('  L1 mean-square error: %.8f\n', err1);
fprintf('  L2 mean-square error: %.8f\n', err2);

% save figure to file
%export_fig('./compare_recErr.png', '-m2.5', '-native');
