function u = l1(P, q, varargin)
%L1 Solves the L1-minimization problem:
%    min || Pu + q ||_1
%with optimization variable u.
%   ================================
%   == EE 236A: HW 8 (Fall 2013)  ==
%   == Name: Eric Kim             ==
%   ================================
%Input:
%   matrix P: MxN
%   vector q: Mx1
%Keyword Parameters:
%   float 'Delta':
%       Specify the Delta parameter used in the sigma barrier parameter
%       computation. Should be greater than 1.0. Defaults to 3.0.
%   float 'eps_feas', 'eps_abs', 'eps_rel':
%       Specify the tolerances for feasibility testing (eps_feas),
%       absolute convergence (eps_abs), or relative convergence (eps_rel).
%       Good default values are: 1e-8, 1e-8, 1e-6.
%   bool 'Verbose':
%       If true, then this will display useful information.
%Output:
%   vector u: Nx1
i_p = inputParser;
i_p.addRequired('P', @isnumeric);
i_p.addRequired('q', @isnumeric);
i_p.addParamValue('Delta', 3.0, @isnumeric);
i_p.addParamValue('eps_feas', 1e-8, @isnumeric);
i_p.addParamValue('eps_abs', 1e-8, @isnumeric);
i_p.addParamValue('eps_rel', 1e-6, @isnumeric);
i_p.addParamValue('verbose', false, @islogical);
i_p.parse(P, q, varargin{:});
P = i_p.Results.P;
q = i_p.Results.q;
DELTA = i_p.Results.Delta;
eps_feas = i_p.Results.eps_feas;
eps_abs = i_p.Results.eps_abs;
eps_rel = i_p.Results.eps_rel;
VERBOSE = i_p.Results.verbose;

if size(q, 1) == 1
    % Enforce that q is a column vector (Mx1)
    q = q';
end

m = size(P, 1);
n = size(P, 2);

% == Initialize data matrices A,b,c
I = eye(m);
A = [[P, -I];
     [-P, -I]];
b = [-q;
      q];
c = [zeros([1,n]), ones([1,m])]';

% ==== 0.) Choose initial s, x, z
% x, s are strictly primal feasible, z strictly dual feasible, AND s > 0, z > 0
[x, s, z] = compute_initial_feasible(A, b, c);

% == Naive starting points: x = [u_ls, 0], s = 1, z = 1.
%u_init = -P\q;
%x = zeros([n+m, 1]);
%x(1:n) = u_init;
%s = ones([2*m, 1]);
%z = ones([2*m, 1]);

if (any(s <= 0))
    disp_ek('WARNING: s is not strictly positive!', VERBOSE);
end
if (any(z <= 0))
    disp_ek('WARNING: z is not strictly positive!', VERBOSE);
end

status = nan;
iter_cur = 1;
% ==== Iterate until convergence
while (1)
    tStart = tic; % Keep track of separate timer to avoid trashing other timers
    disp_ek(sprintf('==== (START Iter %d) ====', iter_cur), VERBOSE);
    % ==== 1.) Check convergence (stopping conditions)
    if (is_converged(A, b, c, x, s, z, eps_feas, eps_abs, eps_rel, iter_cur, VERBOSE))
        u = x(1:n); % Recall: x = [u, v]
        status = 'Converged';
        break;
    end
    % ==== 2.) Update x, s, z
    % == 2.a.) Compute affine scaling direction xdel_a, sdel_a, zdel_a
    r_p = (A*x + s - b);
    r_d = (A'*z + c);
    s_hprod_z = s.*z;     % Hadamard product between s, z
    s_dot_z = s'*z;       % Dot product between s, z
    invS_hprod_z = (1./s).*z; % Hadamard product between (1./s), z
    
    % Optimization: Solve a smaller linear system by exploiting the problem
    % structure. Let xdel = [udel, vdel], solve for udel!
    % Note: Recall that for diag mats D, vec v:    
    %        D*v  ==  v.*d        % d is diag entries of D
    %   inv(D)*v  ==  v./d
    % Let's use these tricks to avoid explicitly constructing these big
    % diag mats diag(s),diag(z) to speed up each iteration (dramatically!).
    Dvec = (1./s).*z;
    D1vec = Dvec(1:m);
    D2vec = Dvec(m+1:end);
    D2_minus_D1 = diag(D2vec - D1vec); % We use this 2x, might as well construct it once
    D_tilde = diag(D1vec + D2vec - ((D2vec-D1vec).^2) .* (1./(D1vec+D2vec)));
    
    lhs = P'*D_tilde*P;     % lhs is shared for 2.a and 2.c
    bz_a = -r_p;
    bx_a = -r_d;
    bs_a = -s_hprod_z;    % Recall: for 2.a. we assume sigma=0, then we estimate sigma
    
    % rhs_a_old is the RHS of the original linear system with xdel_a, but we
    % will break it down into a smaller system for udel_a.
    rhs_a_old = (bx_a + (A'*(bz_a.*invS_hprod_z)) - A'*(bs_a./s));
    b1_a = rhs_a_old(1:n);
    b2_a = rhs_a_old(n+1:end);
    rhs_a = b1_a - P'*(b2_a.*((D2vec - D1vec).*(1./(D1vec+D2vec))));
    udel_a = lhs\rhs_a;
    vdel_a = (b2_a - D2_minus_D1*P*udel_a)./(D1vec+D2vec);
    xdel_a = [udel_a; vdel_a];
    
    %sdel_a = bz_a - A*xdel_a; % This works too, but let's use the lec. notes version for kicks
    zdel_a = (bs_a - (bz_a - A*xdel_a).*z)./s; % Recall: inv(S)*b = b./s
    sdel_a = (bs_a - s.*zdel_a)./z;
  
    % == 2.b.) Compute barrier parameter sigma
    % First, compute how "far" we can go with (zdel, xdel, sdel)
    alpha_pa = compute_alpha(s, sdel_a);
    alpha_da = compute_alpha(z, zdel_a);
    sigma = (((s + alpha_pa*sdel_a)'*(z + alpha_da*zdel_a)) / s_dot_z)^DELTA;

    % == 2.c.) Compute actual search direction xdel, sdel, zdel
    bs = ((sigma*(s_dot_z)/m)*ones([2*m, 1])) - s_hprod_z; % bz, bx stay the same as bz_a, bx_a
    % rhs_old is the RHS of the original linear system with xdel, but we
    % will break it down into a smaller system for udel.
    rhs_old = (bx_a + (A'*(bz_a.*invS_hprod_z)) - A'*(bs./s));
    b1 = rhs_old(1:n);
    b2 = rhs_old(n+1:end);
    rhs = b1 - P'*(b2.*((D2vec - D1vec).*(1./(D1vec+D2vec))));
    udel = lhs\rhs;
    vdel = (b2 - D2_minus_D1*P*udel)./(D1vec+D2vec);
    xdel = [udel; vdel];
    zdel = (bs - (bz_a - A*xdel).*z)./s;
    sdel = (bs - s.*zdel)./z;

    alpha_p = compute_alpha(s, sdel);
    alpha_d = compute_alpha(z, zdel);
    % == 2.d.) Update x, z, s
    step_x = min([1.0, 0.99*alpha_p]) * xdel;
    step_s = min([1.0, 0.99*alpha_p]) * sdel;
    step_z = min([1.0, 0.99*alpha_d]) * zdel;
    
    x = x + step_x;
    s = s + step_s;
    z = z + step_z;
    if any(s < 0)
        disp_ek('WARNING: s is not strictly positive!', VERBOSE);
    elseif any(z < 0)
        disp_ek('WARNING: z is not strictly positive!', VERBOSE);
    end        
    dur_iter = toc(tStart);
    disp_ek(sprintf('[iter: %d] ||r_p||: %.4f ||r_d||: %.4f s*z: %.4f (%.2f secs)', iter_cur, norm(r_p), norm(r_d), s'*z, dur_iter), true);
    disp_ek(sprintf('==== (END Iter %d) ====', iter_cur), VERBOSE);    
    iter_cur = iter_cur + 1;
end % ends while(1)

disp(sprintf('==== HALT, %d iters. Status=%s', iter_cur, status));
end

function alpha_out = compute_alpha(s, sdel)
%COMPUTE_ALPHA computes the "farthest" alpha can be in direction sdel
%such that (s+alpha*sdel) >= 0.
%NOTE: Rarely, cvx (or Matlab) explodes with a mysterious error message:
%          Undefined function 'variable' for input arguments of type 'char'.
%          Error in l1>compute_alpha (line 183)
%               variable alpha_ek(1)
%          Error in l1 (line 133)
%               alpha_pa = compute_alpha(s, sdel_a);
%      Strange. Re-running usually fixes it.
cvx_begin quiet
    variable alpha_ek(1)
    maximize( alpha_ek )
    subject to
        s + (alpha_ek * sdel) >= 0
        0.0 <= alpha_ek <= 1.0
cvx_end
alpha_out = alpha_ek;
end

function out = is_converged(A, b, c, x, s, z, eps_feas, eps_abs, eps_rel, cur_iter, VERBOSE)
%IS_CONVERGED Returns true if our current x, s, z have achieved:
%   1.) Feasibility (eps_feas)
%   2.) Absolute Convergence (eps_abs) or Relative Convergence (eps_rel)
r_p = A*x + s - b;      % Primal residual
r_d = A'*z + c;         % Dual residual
% Check feasibility first
is_feas_p = norm(r_p) <= eps_feas*max([1.0, norm(b)]);
is_feas_d = norm(r_d) <= eps_feas*max([1.0, norm(c)]);
if (~is_feas_p || ~is_feas_d)
    % Infeasible solutions
    out = false;
    disp_ek(sprintf('== Infeasible! Primal: %d Dual: %d', is_feas_p, is_feas_d), VERBOSE);
    return;
end
% Are feasible, check absolute convergence
disp_ek('== Feasible! Checking convergence...', VERBOSE);
if ((s'*z) <= eps_abs)
    out = true;
    disp('== Absolute convergence!');
    return;
end
% Last chance, check relative convergence!
if ((((-b'*z) > 0) && (((s'*z)/(-b'*z)) <= eps_rel)) || ...
    (((c'*x) < 0) && (((s'*z)/(-c'*x)) <= eps_rel)))
    out = true;
    disp('== Relative convergence!');
    return;
end
% Not converged.
out = false;
disp_ek('== Not converged yet!', VERBOSE);
end

function [x_feas, s_feas, z_feas] = compute_initial_feasible(A, b, c)
%COMPUTE_FEASIBLE Computes feasible primal (x,s) and dual (z) starting
%points. In addition, s, z must be strictly positive (s > 0, z > 0).
m = size(A,1) / 2;
n = size(A,2) - m;
P = A(1:m, 1:n);
q = -b(1:m);
% 1.) Primal feasible point via the L2-norm solution
u_ls = P\(-q);
u0 = u_ls;
% Construct a v0 from u0 such that [u0, v0] is strictly primal feasible.
v0 = zeros([m, 1]);
r_ls = P*u0 + q;
for i=1:m
    v0(i) = max([r_ls(i), -r_ls(i)]) + 0.001; % add 0.001 for strict feasibility
end
x_feas = [u0; v0];    
s_feas = b - A*x_feas; % recall: Ax+s=b => s=b-Ax
% 2.) Dual feasible point
% Construct a z from r_ls s.t. z is feasible AND strictly positive.
maxabs_rls = max(abs(r_ls));
tau = (0.5-0.001) / maxabs_rls; % Subtract 0.001 for strict dual feasibility
gamma_1 = (1/2)*ones([m,1]) + tau*r_ls;
gamma_2 = (1/2)*ones([m,1]) - tau*r_ls;
z_feas = [gamma_1; gamma_2];
end

function disp_ek(s, VERBOSE)
%DISP_EK Display input string only if VERBOSE is true.
if VERBOSE
    disp(s);
end
end
